<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Get customer ID and amount from URL parameters
$customerId = $_GET['customer_id'] ?? null;
$amount = $_GET['amount'] ?? 0;

if (!$customerId) {
    header('Location: ../auth/login.php');
    exit();
}

// Get customer details
$stmt = $pdo->prepare("SELECT * FROM customers WHERE id = ?");
$stmt->execute([$customerId]);
$customer = $stmt->fetch();

if (!$customer) {
    die('Customer not found');
}

// Get store settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'currency')");
$stmt->execute();
$settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$storeName = $settings['store_name'] ?? 'Store';
$storeAddress = $settings['store_address'] ?? '';
$storePhone = $settings['store_phone'] ?? '';
$storeEmail = $settings['store_email'] ?? '';
$currency = $settings['currency'] ?? 'Ksh';

// Get customer's outstanding balance
$stmt = $pdo->prepare("
    SELECT 
        ct.id,
        ct.sale_id,
        ct.amount,
        s.invoice_number,
        s.created_at as sale_date,
        ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
        COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
    FROM credit_transactions ct
    JOIN sales s ON ct.sale_id = s.id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    WHERE ct.customer_id = ?
    GROUP BY ct.id
    HAVING balance > 0
    ORDER BY s.created_at DESC
");
$stmt->execute([$customerId]);
$unpaidInvoices = $stmt->fetchAll();

// Get items for each unpaid invoice
foreach ($unpaidInvoices as &$invoice) {
    $stmt = $pdo->prepare("
        SELECT 
            p.name as product_name,
            si.quantity,
            si.price,
            (si.quantity * si.price) as total
        FROM sale_items si
        JOIN products p ON si.product_id = p.id
        WHERE si.sale_id = ?
        ORDER BY p.name
    ");
    $stmt->execute([$invoice['sale_id']]);
    $invoice['items'] = $stmt->fetchAll();
}

$totalBalance = array_sum(array_column($unpaidInvoices, 'balance'));

// Get M-Pesa settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'mpesa_settings'");
$stmt->execute();
$mpesaSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
$mpesaEnabled = ($mpesaSettings['enable_mpesa'] ?? '0') === '1';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Make Payment - <?php echo htmlspecialchars($storeName); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .payment-container {
            max-width: 800px;
            margin: 50px auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .payment-header {
            background: linear-gradient(135deg, #4299e1 0%, #667eea 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .payment-body {
            padding: 40px;
        }
        .invoice-card {
            border: 1px solid #e2e8f0;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 15px;
            background: #f8fafc;
        }
        .invoice-card.selected {
            border-color: #4299e1;
            background: #ebf8ff;
        }
        .pay-button {
            background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
            border: none;
            padding: 15px 40px;
            border-radius: 50px;
            color: white;
            font-weight: 600;
            font-size: 18px;
            transition: all 0.3s ease;
        }
        .pay-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(72, 187, 120, 0.3);
        }
        .total-amount {
            background: linear-gradient(135deg, #ed8936 0%, #dd6b20 100%);
            color: white;
            padding: 20px;
            border-radius: 15px;
            text-align: center;
            margin: 20px 0;
        }
        .items-list {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 10px;
            border: 1px solid #e9ecef;
        }
        .items-list h6 {
            color: #495057;
            margin-bottom: 10px;
            font-weight: 600;
        }
        .item-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            border-bottom: 1px solid #dee2e6;
        }
        .item-row:last-child {
            border-bottom: none;
        }
        .item-name {
            font-weight: 500;
            color: #333;
        }
        .item-quantity {
            color: #6c757d;
            font-size: 0.9em;
        }
        .item-price {
            font-weight: 600;
            color: #28a745;
        }
    </style>
</head>
<body>
    <div class="payment-container">
        <div class="payment-header">
            <h1><i class="fas fa-credit-card me-3"></i><?php echo htmlspecialchars($storeName); ?></h1>
            <p class="mb-0">Secure Payment Portal</p>
        </div>
        
        <div class="payment-body">
            <div class="row">
                <div class="col-md-8">
                    <h3 class="mb-4">Hello, <?php echo htmlspecialchars($customer['name']); ?>!</h3>
                    
                    <?php if (empty($unpaidInvoices)): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle me-2"></i>
                            Great news! You have no outstanding payments.
                        </div>
                    <?php else: ?>
                        <h5 class="mb-3">Outstanding Invoices</h5>
                        <form id="paymentForm">
                            <input type="hidden" name="customer_id" value="<?php echo $customerId; ?>">
                            
                            <?php foreach ($unpaidInvoices as $invoice): ?>
                                <div class="invoice-card">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="invoice_ids[]" value="<?php echo $invoice['id']; ?>" id="invoice_<?php echo $invoice['id']; ?>" data-balance="<?php echo $invoice['balance']; ?>" checked>
                                        <label class="form-check-label" for="invoice_<?php echo $invoice['id']; ?>">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong>Invoice #<?php echo htmlspecialchars($invoice['invoice_number']); ?></strong><br>
                                                    <small class="text-muted">Date: <?php echo date('M d, Y', strtotime($invoice['sale_date'])); ?></small>
                                                </div>
                                                <div class="text-end">
                                                    <span class="badge bg-danger fs-6"><?php echo $currency; ?> <?php echo number_format($invoice['balance'], 2); ?></span>
                                                </div>
                                            </div>
                                        </label>
                                    </div>
                                    
                                    <?php if (!empty($invoice['items'])): ?>
                                        <div class="items-list">
                                            <h6><i class="fas fa-shopping-cart me-2"></i>Items in this Invoice</h6>
                                            <?php foreach ($invoice['items'] as $item): ?>
                                                <div class="item-row">
                                                    <div>
                                                        <div class="item-name"><?php echo htmlspecialchars($item['product_name']); ?></div>
                                                        <div class="item-quantity">Qty: <?php echo $item['quantity']; ?> × <?php echo $currency; ?> <?php echo number_format($item['price'], 2); ?></div>
                                                    </div>
                                                    <div class="item-price">
                                                        <?php echo $currency; ?> <?php echo number_format($item['total'], 2); ?>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                            
                            <div class="total-amount">
                                <h4>Total Amount: <span id="totalAmount"><?php echo $currency; ?> <?php echo number_format($totalBalance, 2); ?></span></h4>
                            </div>
                            
                            <?php if ($mpesaEnabled): ?>
                                <div class="mb-4">
                                    <label class="form-label">M-Pesa Phone Number</label>
                                    <input type="tel" class="form-control" name="phone" placeholder="254XXXXXXXXX" required>
                                    <small class="text-muted">Enter your M-Pesa registered phone number</small>
                                </div>
                                
                                <div class="text-center">
                                    <button type="submit" class="btn pay-button">
                                        <i class="fas fa-mobile-alt me-2"></i>Pay with M-Pesa
                                    </button>
                                </div>
                                
                                <!-- Footer Section -->
                                <div class="mt-4 text-center">
                                    <small class="text-muted">
                                        <?php ob_start(); include '../includes/footer.php'; $footerContent = ob_get_clean(); echo trim(strip_tags($footerContent)); ?> | +254702214793
                                    </small>
                                </div>
                            <?php else: ?>
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle me-2"></i>
                                    M-Pesa payments are currently disabled. Please contact us to arrange payment.
                                </div>
                                
                                <!-- Footer Section -->
                                <div class="mt-4 text-center">
                                    <small class="text-muted">
                                        <?php ob_start(); include '../includes/footer.php'; $footerContent = ob_get_clean(); echo trim(strip_tags($footerContent)); ?> | +254702214793
                                    </small>
                                </div>
                            <?php endif; ?>
                        </form>
                    <?php endif; ?>
                </div>
                
                <div class="col-md-4">
                    <div class="card">
                        <div class="card-header">
                            <h5><i class="fas fa-store me-2"></i>Contact Information</h5>
                        </div>
                        <div class="card-body">
                            <p><i class="fas fa-phone me-2"></i><?php echo htmlspecialchars($storePhone); ?></p>
                            <p><i class="fas fa-envelope me-2"></i><?php echo htmlspecialchars($storeEmail); ?></p>
                            <p><i class="fas fa-map-marker-alt me-2"></i><?php echo htmlspecialchars($storeAddress); ?></p>
                            
                            <hr>
                            <div class="d-grid gap-2">
                                <a href="tel:<?php echo htmlspecialchars($storePhone); ?>" class="btn btn-outline-primary">
                                    <i class="fas fa-phone me-2"></i>Call Us
                                </a>
                                <a href="mailto:<?php echo htmlspecialchars($storeEmail); ?>" class="btn btn-outline-secondary">
                                    <i class="fas fa-envelope me-2"></i>Email Us
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Update total amount when invoices are selected/deselected
        document.querySelectorAll('input[name="invoice_ids[]"]').forEach(function(checkbox) {
            checkbox.addEventListener('change', function() {
                let total = 0;
                document.querySelectorAll('input[name="invoice_ids[]"]:checked').forEach(function(checked) {
                    total += parseFloat(checked.getAttribute('data-balance'));
                });
                document.getElementById('totalAmount').textContent = '<?php echo $currency; ?> ' + total.toFixed(2);
            });
        });

        // Handle payment form submission
        document.getElementById('paymentForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            
            // Validate selections
            const checkedInvoices = document.querySelectorAll('input[name="invoice_ids[]"]:checked');
            if (checkedInvoices.length === 0) {
                alert('Please select at least one invoice to pay.');
                return;
            }
            
            // Validate phone number
            const phone = formData.get('phone');
            if (!phone || phone.length < 10) {
                alert('Please enter a valid phone number.');
                return;
            }
            
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
            
            // Calculate total amount
            let totalAmount = 0;
            checkedInvoices.forEach(function(invoice) {
                totalAmount += parseFloat(invoice.getAttribute('data-balance'));
            });
            
            // Send M-Pesa STK Push request
            fetch('../api/mpesa_stkpush.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    phone: phone,
                    amount: totalAmount,
                    customer_id: formData.get('customer_id'),
                    customer_name: '<?php echo htmlspecialchars($customer['name']); ?>'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('✅ STK Push sent successfully! Please check your phone to complete the payment.');
                    // Refresh page after 3 seconds
                    setTimeout(() => {
                        window.location.reload();
                    }, 3000);
                } else {
                    alert('❌ Error: ' + (data.message || 'Failed to send STK Push'));
                }
            })
            .catch(error => {
                console.error('Payment Error:', error);
                alert('❌ Network error. Please try again.');
            })
            .finally(() => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            });
        });
    </script>
</body>
</html> 